// ==UserScript==
// @name         5ch サムネイル自動表示_Twitter画像・動画（動画縮小大きめトグル対応）
// @namespace    https://example.com/
// @version      1.2.2
// @description  5ch投稿内のTwitter画像・動画URLを検出し、投稿末尾に縮小表示のサムネイルや動画プレイヤーを自動表示。クリックで拡大・縮小切替。動画は縮小時300px幅または400px高さ基準。ボタン無し。
// @match        *://*.5ch.net/test/read.cgi/*
// @grant        none
// ==/UserScript==

(function () {
    'use strict';

    // CSSスタイル追加
    const style = document.createElement('style');
    style.textContent = `
      .twitter-media-thumbnail {
        cursor: pointer;
        max-width: 150px;
        height: auto;
        transition: max-width 0.3s ease, width 0.3s ease, height 0.3s ease;
        display: block;
        margin-top: 4px;
      }
      .twitter-media-thumbnail.video-thumb {
        /* 最大サイズはJS側で調整するのでここは一旦削除 */
      }
      .twitter-media-thumbnail.expanded {
        max-width: 100% !important;
        width: auto !important;
        height: auto !important;
      }
    `;
    document.head.appendChild(style);

    const THUMBNAIL_CLASS = 'twitter-media-thumbnail';

    /* 画像もサムネイルするならこちら（現在は【5ch サムネイル表示（画像専用）】でツイッター画像も処理）
    // TwitterメディアURLか判定（pbs.twimg.com/media、video.twimg.com/amplify_video、video.twimg.com/ext_tw_video対応）
    function isTwitterMediaUrl(url) {
        return /^https?:\/\/(pbs\.twimg\.com\/media|video\.twimg\.com\/amplify_video|video\.twimg\.com\/ext_tw_video)\/.+\.(jpg|jpeg|png|gif|mp4|mov|webm)(\?.*)?$/i.test(url);
    }
    */

    // 動画だけを対象にし、画像は除外する場合はこちら
    // TwitterメディアURLか判定（pbs.twimg.com/media、video.twimg.com/amplify_video、video.twimg.com/ext_tw_video対応）
    function isTwitterMediaUrl(url) {
        // メディアURL判定
        const isMedia = /^https?:\/\/(pbs\.twimg\.com\/media|video\.twimg\.com\/amplify_video|video\.twimg\.com\/ext_tw_video)\/.+\.(jpg|jpeg|png|gif|mp4|mov|webm)(\?.*)?$/i.test(url);

        // 画像ファイルのURLを除外
        if (isMedia && /\.(jpg|jpeg|png|gif)$/i.test(url)) {
            return false; // 画像URLは除外
        }
        return isMedia; // 画像以外のメディアURLはtrue
    }

    // クリックで拡大・縮小トグル
    function toggleExpand(event) {
        event.currentTarget.classList.toggle('expanded');
        if(event.currentTarget.classList.contains('expanded')){
            // 拡大時は幅・高さを自動に戻す
            event.currentTarget.style.width = 'auto';
            event.currentTarget.style.height = 'auto';
        } else {
            // 縮小時は再度サイズ調整をトリガーするためloadedmetadataを手動で発火できないので
            // ここは動画のnaturalサイズを使って再計算する関数を呼び出す形にする
            if(event.currentTarget.tagName === 'VIDEO'){
                adjustVideoSize(event.currentTarget);
            }
        }
    }

    // 動画のサイズを幅300pxまたは高さ400pxを最大にして調整
    function adjustVideoSize(video) {
        const MAX_WIDTH = 300;
        const MAX_HEIGHT = 250;

        // videoWidth, videoHeight が未取得の場合はloadedmetadata待ち
        if(video.videoWidth === 0 || video.videoHeight === 0){
            video.addEventListener('loadedmetadata', () => {
                adjustVideoSize(video);
            }, { once: true });
            return;
        }

        const w = video.videoWidth;
        const h = video.videoHeight;
        const maxDim = Math.max(w, h);

        if (maxDim === w) {
            // 幅が大きい → 幅をMAX_WIDTHに合わせて縮小
            const scale = MAX_WIDTH / w;
            video.style.width = `${w * scale}px`;
            video.style.height = `${h * scale}px`;
        } else {
            // 高さが大きい → 高さをMAX_HEIGHTに合わせて縮小
            const scale = MAX_HEIGHT / h;
            video.style.width = `${w * scale}px`;
            video.style.height = `${h * scale}px`;
        }
    }

    // メディアサムネイルを投稿末尾に追加
    async function appendMediaThumbnails(post) {
        if (post.dataset.twitterMediaProcessed) return;
        post.dataset.twitterMediaProcessed = 'true';

        const anchors = post.querySelectorAll('a[href]');
        let appended = false;

        for (const a of anchors) {
            const href = a.href;
            if (isTwitterMediaUrl(href)) {
                const container = document.createElement('div');
                container.style.marginTop = '8px';

                if (href.match(/\.(mp4|mov|webm)/i)) {
                    // 動画
                    const video = document.createElement('video');
                    video.src = href;
                    video.controls = true;
                    video.classList.add(THUMBNAIL_CLASS, 'video-thumb');
                    video.style.height = 'auto';

                    video.addEventListener('click', toggleExpand);

                    // サイズ調整
                    adjustVideoSize(video);

                    container.appendChild(video);
                } else {
                    // 画像
                    const img = document.createElement('img');
                    img.src = href;
                    img.classList.add(THUMBNAIL_CLASS);
                    img.style.height = 'auto';
                    img.addEventListener('click', toggleExpand);
                    container.appendChild(img);
                }

                post.appendChild(container);
                appended = true;
            }
        }

        if (appended) {
            console.log('Twitterメディアのサムネイルを投稿末尾に追加しました');
        }
    }

    // 投稿の処理（メディア検出＆追加）
    async function processPostContent(post) {
        await appendMediaThumbnails(post);
    }

    // 初期処理
    document.querySelectorAll('.post-content').forEach(post => {
        processPostContent(post);
    });

    // 1.5秒後に再処理（動的ロード対応）
    setTimeout(() => {
        document.querySelectorAll('.post-content').forEach(post => {
            processPostContent(post);
        });
    }, 1500);

    // 監視（新規投稿があったら処理）
    const observer = new MutationObserver(mutations => {
        for (const mutation of mutations) {
            for (const node of mutation.addedNodes) {
                if (node.nodeType !== 1) continue;
                if (node.classList.contains('post-content')) {
                    processPostContent(node);
                } else {
                    node.querySelectorAll?.('.post-content').forEach(child => {
                        processPostContent(child);
                    });
                }
            }
        }
    });

    observer.observe(document.body, {
        childList: true,
        subtree: true
    });
})();
